<?php namespace App\Http\Controllers;

use App\Libraries\Utils;
use App\Users;
use Auth;
use Illuminate\Contracts\Auth\Guard;
use Illuminate\Http\Request;
use Input;
use Mail;
use Session;
use URL;
use Validator;

class AuthController extends Controller
{

    public function __construct(Guard $auth)
    {
        $this->auth = $auth;

    }

    public function getLogin()
    {
        return view('admin.login');
    }

    public function postLogin(Request $request)
    {

        $v = Validator::make(
            ['email' => $request->input('email'), 'password' => $request->input('password')],
            ['email' => 'required|email', 'password' => 'required']
        );

        if ($v->fails()) {
            Session::flash('error_msg', Utils::messages($v));
            return redirect()->back()->withInput($request->only('email', 'remember'));
        }

        $credentials = $request->only('email', 'password');

        if ($this->auth->attempt($credentials, $request->has('remember'))) {

            if (Utils::isAdmin(Auth::user()->id)) {
                return redirect()->intended('/admin');
            }

            if (Utils::isCustomer(Auth::user()->id)) {
                return redirect()->intended('/customer');
            }


            if (Utils::isPublisher(Auth::user()->id)) {
                return redirect()->intended('/publisher');
            }

        }

        Session::flash('error_msg', 'Invalid login please try again');

        return redirect()->back()->withInput($request->only('email', 'remember'));
    }

    public function getForgotPassword()
    {
        return view("admin.forgot-password");
    }

    public function postForgotPassword()
    {
        $email = Input::get("email");

        $user = Users::where('email', $email)->first();

        if (sizeof($user) <= 0) {
            Session::flash("error_msg", "An account was not found with that e-mail address. Please input another and try again!");
            return redirect()->back();
        } else {
            $reset_code = Utils::generateResetCode();
            $user->reset_password_code = $reset_code;
            $user->reset_requested_on = \Carbon\Carbon::now();
            $user->save();

            $email_arr = ['name' => $user->name,
                'reset_url' => URL::to('/') . "/reset_password/" . $user->email . "/" . $user->reset_password_code,
                'email' => $user->email];

            Mail::send('emails.reset_password', $email_arr, function ($message) use ($user) {
                $message->to($user->email, $user->name)->subject('RSS - Reset Password Request');
            });

            Session::flash('success_msg', 'Please click on the link we sent to your email to reset password');
            return redirect()->to('/forgot-password');
        }

    }

    public function getReset($email, $code)
    {

        if (strlen($email) <= 0 || strlen($code) <= 0) {
            Session::flash("error_msg", "Invalid Request. Please reset your password.");
            return redirect()->to('/forgot-password');
        }

        //Check code and email
        $user = Users::where('email', $email)->where('reset_password_code', $code)->first();

        if (sizeof($user) <= 0) {
            Session::flash("error_msg", "Invalid Request. Please reset your password.");
            return redirect()->to('/forgot-password');
        } else {
            //check for 24 hrs for token
            $reset_requested_on = \Carbon\Carbon::createFromFormat('Y-m-d G:i:s', $user->reset_requested_on);
            $present_day = \Carbon\Carbon::now();

            if ($reset_requested_on->addDay() > $present_day) {
                //Show new password view
                return view('admin.reset-password', ['email' => $email, 'code' => $code]);
            } else {
                Session::flash("error_msg", "Your password change token has expired. Please reset your password.");
                return redirect()->to('/forgot-password');
            }
        }
    }

    public function postReset()
    {

        $password = Input::get('password', '');
        $password_confirmation = Input::get('password_confirmation', '');

        if ($password == $password_confirmation) {

            $validate_reset = Users::where('email', Input::get('email', ''))->where('reset_password_code', Input::get('code', ''))->first();

            if (sizeof($validate_reset) > 0) {
                $user = Users::where('email', Input::get('email'))->first();
                $user->password = \Hash::make($password);
                $user->save();

                Session::flash('success_msg', 'Your password was changed successfully.');
                return redirect()->to('/login');
            } else {
                Session::flash('error_msg', 'An invalid password was entered.');
                return redirect()->back();
            }
        } else {
            Session::flash('error_msg', 'Both your new password and your password confirmation must be the same.');
            return redirect()->back();
        }
    }

    public function getLogout()
    {
        $this->auth->logout();

        return redirect('/login');
    }


}
